<?php
require_once '../config.php';

if(!isLoggedIn()) {
    echo '<div class="text-center p-8"><p class="text-red-600">Please login first</p></div>';
    exit();
}

$referred_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$user_id = $_SESSION['user_id'];

if(!$referred_id) {
    echo '<div class="text-center p-8"><p class="text-red-600">Referral not found</p></div>';
    exit();
}

// Get referral details
$stmt = $conn->prepare("
    SELECT r.*, u.username, u.email, u.is_active, u.balance, u.created_at as user_created,
           u.telegram_username, u.whatsapp_number, u.referral_code as user_referral_code
    FROM referrals r
    JOIN users u ON r.referred_id = u.id
    WHERE r.referred_id = ? AND r.referrer_id = ?
");
$stmt->bind_param("ii", $referred_id, $user_id);
$stmt->execute();
$result = $stmt->get_result();

if($result->num_rows == 0) {
    echo '<div class="text-center p-8"><p class="text-red-600">Referral not found</p></div>';
    exit();
}

$referral = $result->fetch_assoc();

// Get admin settings
$settings = getSettings();
$referral_percentage = $settings['referral_percentage'];

// Calculate potential earnings
$activation_fee = $settings['activation_fee'];
$potential_earning = $activation_fee * ($referral_percentage / 100);

// Format status
$status_color = $referral['status'] == 'paid' ? 'green' : 'yellow';
$status_icon = $referral['status'] == 'paid' ? 'fa-check-circle' : 'fa-clock';
?>

<div class="space-y-6">
    <!-- Referral Header -->
    <div class="text-center">
        <div class="w-20 h-20 bg-gradient-to-r from-blue-500 to-purple-500 rounded-full flex items-center justify-center text-white font-bold text-2xl mx-auto mb-4">
            <?php echo strtoupper(substr($referral['username'], 0, 1)); ?>
        </div>
        <h3 class="text-2xl font-bold text-gray-800"><?php echo $referral['username']; ?></h3>
        <p class="text-gray-600">Referred User Details</p>
    </div>
    
    <!-- User Information -->
    <div class="bg-gray-50 rounded-lg p-6">
        <h4 class="font-bold text-gray-800 mb-4">
            <i class="fas fa-user mr-2"></i> User Information
        </h4>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
                <div class="mb-3">
                    <p class="text-sm text-gray-600">Username</p>
                    <p class="font-medium"><?php echo $referral['username']; ?></p>
                </div>
                <div class="mb-3">
                    <p class="text-sm text-gray-600">Email</p>
                    <p class="font-medium"><?php echo $referral['email']; ?></p>
                </div>
                <div class="mb-3">
                    <p class="text-sm text-gray-600">Account Status</p>
                    <p class="font-medium">
                        <span class="<?php echo $referral['is_active'] ? 'text-green-600' : 'text-red-600'; ?>">
                            <?php echo $referral['is_active'] ? 'Active' : 'Inactive'; ?>
                        </span>
                    </p>
                </div>
            </div>
            
            <div>
                <div class="mb-3">
                    <p class="text-sm text-gray-600">Joined Date</p>
                    <p class="font-medium"><?php echo date('F d, Y', strtotime($referral['user_created'])); ?></p>
                </div>
                <div class="mb-3">
                    <p class="text-sm text-gray-600">Current Balance</p>
                    <p class="font-medium">৳<?php echo number_format($referral['balance'], 2); ?></p>
                </div>
                <div class="mb-3">
                    <p class="text-sm text-gray-600">Referral Code</p>
                    <p class="font-medium font-mono"><?php echo $referral['user_referral_code']; ?></p>
                </div>
            </div>
        </div>
        
        <!-- Contact Information -->
        <?php if($referral['telegram_username'] || $referral['whatsapp_number']): ?>
        <div class="mt-4 pt-4 border-t">
            <h5 class="font-medium text-gray-800 mb-2">Contact Information</h5>
            <div class="flex flex-wrap gap-3">
                <?php if($referral['telegram_username']): ?>
                <a href="https://t.me/<?php echo ltrim($referral['telegram_username'], '@'); ?>" 
                   target="_blank" 
                   class="inline-flex items-center px-3 py-1 bg-blue-100 text-blue-700 rounded text-sm hover:bg-blue-200 transition duration-300">
                    <i class="fab fa-telegram mr-1"></i> <?php echo $referral['telegram_username']; ?>
                </a>
                <?php endif; ?>
                
                <?php if($referral['whatsapp_number']): ?>
                <a href="https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $referral['whatsapp_number']); ?>" 
                   target="_blank" 
                   class="inline-flex items-center px-3 py-1 bg-green-100 text-green-700 rounded text-sm hover:bg-green-200 transition duration-300">
                    <i class="fab fa-whatsapp mr-1"></i> <?php echo $referral['whatsapp_number']; ?>
                </a>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
    
    <!-- Referral Information -->
    <div class="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg p-6">
        <h4 class="font-bold text-gray-800 mb-4">
            <i class="fas fa-handshake mr-2"></i> Referral Information
        </h4>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
                <div class="mb-4">
                    <p class="text-sm text-gray-600">Referral Status</p>
                    <div class="flex items-center mt-1">
                        <span class="px-3 py-1 inline-flex text-sm leading-5 font-semibold rounded-full bg-<?php echo $status_color; ?>-100 text-<?php echo $status_color; ?>-800">
                            <i class="fas <?php echo $status_icon; ?> mr-1"></i>
                            <?php echo ucfirst($referral['status']); ?>
                        </span>
                    </div>
                </div>
                
                <div class="mb-4">
                    <p class="text-sm text-gray-600">Commission Earned</p>
                    <p class="text-2xl font-bold text-green-600">
                        ৳<?php echo number_format($referral['bonus_amount'] ?? 0, 2); ?>
                    </p>
                </div>
                
                <div class="mb-4">
                    <p class="text-sm text-gray-600">Referral Date</p>
                    <p class="font-medium"><?php echo date('F d, Y h:i A', strtotime($referral['created_at'])); ?></p>
                </div>
            </div>
            
            <div>
                <div class="mb-4">
                    <p class="text-sm text-gray-600">Commission Rate</p>
                    <p class="text-xl font-bold text-blue-600"><?php echo $referral_percentage; ?>%</p>
                </div>
                
                <div class="mb-4">
                    <p class="text-sm text-gray-600">Activation Fee</p>
                    <p class="font-medium">৳<?php echo number_format($activation_fee, 2); ?></p>
                </div>
                
                <?php if(!$referral['is_active']): ?>
                <div class="mb-4">
                    <p class="text-sm text-gray-600">Potential Earnings</p>
                    <p class="font-medium text-green-600">৳<?php echo number_format($potential_earning, 2); ?></p>
                    <p class="text-xs text-gray-500">When user activates account</p>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Actions -->
    <div class="pt-4 border-t">
        <div class="flex justify-center space-x-3">
            <?php if($referral['telegram_username']): ?>
            <a href="https://t.me/<?php echo ltrim($referral['telegram_username'], '@'); ?>" 
               target="_blank" 
               class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-2 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                <i class="fab fa-telegram mr-2"></i> Message on Telegram
            </a>
            <?php endif; ?>
            
            <?php if($referral['whatsapp_number']): ?>
            <a href="https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $referral['whatsapp_number']); ?>" 
               target="_blank" 
               class="bg-gradient-to-r from-green-600 to-emerald-700 text-white font-bold py-2 px-6 rounded-lg hover:from-green-700 hover:to-emerald-800 transition duration-300">
                <i class="fab fa-whatsapp mr-2"></i> Message on WhatsApp
            </a>
            <?php endif; ?>
            
            <button onclick="closeReferralModal()" 
                    class="bg-gray-200 text-gray-700 font-bold py-2 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                <i class="fas fa-times mr-2"></i> Close
            </button>
        </div>
    </div>
</div>

<script>
function closeReferralModal() {
    document.getElementById('referralModal').classList.add('hidden');
    document.getElementById('referralModal').classList.remove('flex');
}
</script>